/* Adam Ryan, Michael Henderson 2007
 * This program is a project for the REU Site 2007 research program. 
 * This code was originally adapted from the book "Multicast Sockets: Practical Guide for Programmers" 
 * by David Makofske and Kevin Almeroth
 */

using System;             //For Console and Exception classes
using System.Net;         //For IPAddress class
using System.Net.Sockets; //For Socket class
using System.Diagnostics; //For Diagnostics class
using System.IO;          //For IO class
using System.Threading;   //For Threading class

/// <summary>
/// Multicast Sender Class
/// Version 1.0.0 Beta v3
/// Allows for sending of text strings to multicast IP addresses and the receiving of the pseudo NACKs via Thread
/// </summary>

class MulticastSender
{
    const int Min_Port = 1024;     //minimum port value
    const int Max_Port = 65534;    //maximum port value

    public static void Main(string[] args)
    {
        Boolean    done = false;   //loop variable
        int        mcastPort;      //destination port
        int        ttl;            //Time To Live (1 hop; 0 hops: LAN only)--user input
        int        pCount = 0;     //Sequence for checking if the receiver has received the data 
        IPAddress  mcastIP;        //destination multicast address
        IPEndPoint iPEP;           //IP endpoint
        Socket     mcastSock;      //multicast socket
        Stopwatch  clock;          //Keeps time
        string     now;            //Shows current time 
        TextWriter tw;             //For writing text to a file
        Thread     receiver;       //For receiving multiple ACK messages 
        MulticastSender ms = new MulticastSender(); //Solely to enable use of the packetReceive method
        
        //verify correct usage of command line arguments; there should be three. Tell user what they should
        //be if they are invalid
        if (args.Length != 3)
        {
            Console.Error.WriteLine("Usage: MCSend " + "<Multicast IP> <Multicast Port> <TTL>");
            return;
        }

        //validate the input multicast IP address
        try
        {
            mcastIP = IPAddress.Parse(args[0]);
        }
        catch (Exception)
        {
            Console.Error.WriteLine("The IP address you have entered is invalid.");
            return;
        }

        //Validate the input port number.  tell the user the correct range if outside of it
        try
        {
            mcastPort = Int32.Parse(args[1]);
        }
        catch (Exception)
        {
            Console.Error.WriteLine("Invalid Port Specified.");
            return;
        }
        if ((mcastPort < Min_Port) || (mcastPort > Max_Port))
        {
            Console.Error.WriteLine("That port is not in the correct range.");
            Console.Error.WriteLine("The port must be between " + Min_Port + " and " + Max_Port);
            return;
        }
        
        //Validate the Time To Live. must be in the range 0-255 inclusive. inform user of this if need be.
        try
        {
            ttl = Int32.Parse(args[2]);
        }
        catch (Exception)
        {
            Console.Error.WriteLine("Invalid TTL Specified.");
            return;
        }
        if ((ttl < 0) || (ttl > 255)) //Makes sure ther TTL is within the appropriate range
        {
            Console.Error.WriteLine("That TTL is not in the correct range.");
            Console.Error.WriteLine("The TTL must be between 0 and 255.");
            return;
        }
                       
        //create an IP endpoint class instance for the multicast group on the multicast port
        iPEP = new IPEndPoint(mcastIP, mcastPort);

        try
        {
            //Creates clock
            clock = new Stopwatch();
            clock.Reset();

            //Intializes Textwriter
            tw = new StreamWriter("C:\\mc_send_log.txt", true);

            //Creates the socket
            mcastSock = new Socket(AddressFamily.InterNetwork, SocketType.Dgram, ProtocolType.Udp);

            //sets the time to live.
            mcastSock.SetSocketOption(SocketOptionLevel.IP, SocketOptionName.MulticastTimeToLive, ttl);

            //Delegate for the packetReceive method (allows for the passing of arguments)
            ThreadStart starter = delegate { ms.packetReceive(mcastSock, tw); };

            //Initializes and starts the Thread
            receiver = new Thread(starter);
            receiver.IsBackground = true; //Runs the thread in the background
            receiver.Start();


            //Notifies the user that they are connected and ready to go; Instructs them to begin typing and sending
            Console.WriteLine("Set to multicast to {0} on port {1}.", mcastIP, mcastPort);
            Console.WriteLine("Begin typing.  " + "[Enter/Return] to send line, [Ctrl]-[C] to quit:\n");

            //Time mark for a new entry into the log
            tw.WriteLine(DateTime.Now + "\r\n");

            //loop for sending, etc.
            while (!done)
            {
                //reads user input from the terminal
                string strInput = Console.ReadLine() + ":" + pCount;
                tw.WriteLine(strInput);

                //Starts the clock after the user input
                clock.Start();

                //Updates the "now" string
                now = DateTime.Now.Hour + ":" + DateTime.Now.Minute + ":" + DateTime.Now.Second + ":" + DateTime.Now.Millisecond;

                //formats the input from the terminal into an array of bytes so that it can be sent
                System.Text.ASCIIEncoding encode = new System.Text.ASCIIEncoding();
                byte[] senderData = encode.GetBytes(strInput);

                //sends the array of bytes as a data packet
                mcastSock.SendTo(senderData, 0, senderData.Length, SocketFlags.None, iPEP);

                //stop the clock
                clock.Stop();

                //converts the time from ticks to nanoseconds.  Recall 1 ns = 1 billionth of a second
                long totalTime = clock.ElapsedTicks;

                //notifies the sender that the line was sent
                string message = "<Sent " + senderData.Length + " bytes in " + totalTime + " ticks at " + now + ">";
                Console.WriteLine(message + "\r\n");
                tw.WriteLine(message);

                //reset clock back to 0
                clock.Reset();

                //Updates the counter
                pCount++;
            }

            //Abort thread
            receiver.Abort();

            //Creates a break to differentiate between different log entries
            tw.WriteLine("------------END OF TRANSMISSION------------");
            tw.Flush();

            //closes TextWriter
            tw.Close();

            //closes the socket
            mcastSock.Close();
        }

        //basic error handling
        catch (SocketException se)
        {
            Console.Error.WriteLine("Socket Exception: " + se.ToString());
            return;
        }
        catch (IOException ioe)
        {
            Console.Error.WriteLine("IO Exception: " + ioe.ToString());
            return;
        }
        catch (ThreadAbortException tae)
        {
            Console.Error.WriteLine("Thread Abort Exception: " + tae.ToString());
            return;
        }
        catch (ThreadInterruptedException tie)
        {
            Console.Error.WriteLine("Thread Interrupted Exception: " + tie.ToString());
            return;
        }
        catch (Exception e)
        {
            Console.Error.WriteLine("Exception: " + e.ToString());
            return;
        }
        return;
    }

    //This was originally in the while loop, but was taken out due to a bug where the system would only
    //receive the first ACK message that came to it. The method below, which implements threading, replaces
    //that code. *Mike*
    void packetReceive(Socket sock, TextWriter tw)
    {
        try
        {
            //creates the endpoint class--this one is used for receiving the ACK
            IPEndPoint receivePt = new IPEndPoint(IPAddress.Any, 65535);
            EndPoint tempReceivePt = (EndPoint)receivePt;

            //Decodes the ACK message and puts it into the stringData variable
            System.Text.ASCIIEncoding coder = new System.Text.ASCIIEncoding();
            byte[] data = new byte[1024];
            string stringData;
            string line;
            int recv;

            //Binds the EndPoint to the socket
            sock.Bind(tempReceivePt);

            //Take in all the ACK messages that all the other subscribers are sending, write it out to the
            //console and update the log using the same string of data
            while (true)
            {
                recv = sock.ReceiveFrom(data, ref tempReceivePt);
                stringData = coder.GetString(data, 0, recv);
                line = "<Message from " + tempReceivePt.ToString() + ": " + stringData + ">\r\n";
                Console.WriteLine(line);
                tw.WriteLine(line);
                tw.Flush();
            }
        }
        catch (SocketException se)
        {
            Console.Error.WriteLine("Socket Exception: " + se.ToString());
            return;
        }
        catch (Exception e)
        {
            Console.Error.WriteLine("Exception: " + e.ToString());
            return;
        }
    }
}